/** @file   route.cpp
 * @brief   Implementation of Route class
 * @version $Revision: 1.2 $
 * @date    $Date: 2006/03/29 22:28:34 $
 * @author  Tomi Lamminsaari
 */


#include "route.h" // class's header file
#include "www_assert.h"

using std::string;
using std::ifstream;
using std::istream;


namespace WeWantWar {



///
/// Constructor, destructor
/// =======================

/** Constructor
 */
Route::Route( const string& routeName ) :
  m_name( routeName )
{
	
}

/** Copy constructor
 */
Route::Route( const Route& rRoute ) :
  m_waypoints( rRoute.m_waypoints ),
  m_name( rRoute.m_name )
{
}

/** Destructor.
 */
Route::~Route()
{
	
}



/** Assignment operator
 */
Route& Route::operator = ( const Route& rRoute )
{
  if ( this != &rRoute ) {
    m_waypoints = rRoute.m_waypoints;
    m_name = rRoute.m_name;
  }
  return *this;
}



///
/// Methods
/// =======

/** Adds new node to this route.
 */
void Route::addWaypoint( const Waypoint& rN )
{
  m_waypoints.push_back( rN );
}



/** Changes the index'th node
 */
void Route::setWaypoint( int index, const Waypoint& rN )
{
  if ( index < 0 || index >= this->nodeCount() ) {
    return;
  }
  m_waypoints.at( index ) = rN;
}



/** Loads the route from given file
 */
int Route::loadRoute( const string& routefile )
{
  ifstream fin( routefile.c_str() );
  if ( !fin ) {
    return -1;
  }
  string tmp;
  fin >> tmp;
  if ( tmp != "<wewantwar_route>" ) {
    fin.close();
    return -1;
  }
  int ret = this->readRoute( fin );
  fin.close();
  return 0;
}



/** Reads the route from given stream.
 */
int Route::readRoute( istream& rIn )
{
  this->clear();
  
  string tmp;
  while ( true ) {
    if ( rIn.eof() == true ) {
      return -1;
    }
    
    rIn >> tmp;
    if ( tmp == "</wewantwar_route>" ) {
      // We've read the whole route.
      break;
      
    } else if ( tmp == "#" ) {
      // A comment line.
      rIn.ignore( 4096, '\n' );
      
    } else if ( tmp == "<node>" ) {
      if ( this->readNode( rIn ) != 0 ) {
        return -1;
      }
      
    } else if ( tmp == "<blocknode>" ) {
      if ( this->readBlockNode( rIn ) != 0 ) {
        return -1;
      }
      
    } else if ( tmp == "name:" ) {
      rIn >> m_name;
      
    }
  }
  return 0;
}



/** Clears all the waypoint nodes this route has.
 */
void Route::clear()
{
  m_waypoints.clear();
}



///
/// Getter methods
/// ==============

/** Returns the name of this route
 */
string Route::name() const
{
  return m_name;
}



/** Returns the number of nodes there are.
 */
int Route::nodeCount() const
{
  return m_waypoints.size();
}



/** Returns the node
 */
const Route::Waypoint& Route::getWaypoint( int index ) const
{
  WWW_ASSERT( index >= 0 );
  WWW_ASSERT( index < this->nodeCount() );
  return m_waypoints.at( index );
}



/** Returns the distance from given point to index'th waypoint
 */
float Route::distanceFromWaypoint( const eng2d::Vec2D& rPos, int index ) const
{
  WWW_ASSERT( index >= 0 );
  WWW_ASSERT( index < this->nodeCount() );
  
  eng2d::Vec2D tmpV( m_waypoints.at(index).pos );
  tmpV -= rPos;
  return tmpV.length();
}



/** Returns the position of requested waypoint.
 */
eng2d::Vec2D Route::getWaypointPos( int index ) const
{
  WWW_ASSERT( index >= 0 );
  WWW_ASSERT( index < this->nodeCount() );
  
  return m_waypoints.at(index).pos;
}




///
/// Protected methods
/// =================

/** Reads the node from stream
 */
int Route::readNode( istream& rIn )
{
  string tmp1;
  string tmp2;
  
  rIn >> tmp1 >> tmp2;
  Waypoint wp;
  wp.pos = eng2d::Vec2D( atoi(tmp1.c_str()), atoi(tmp2.c_str()) );
  
  this->addWaypoint( wp );
  return 0;
}

int Route::readBlockNode( istream& aIn )
{
  string tmp1;
  string tmp2;
  aIn >> tmp1 >> tmp2;
  Waypoint wp;
  wp.pos = eng2d::Vec2D( atoi(tmp1.c_str())*32, atoi(tmp2.c_str())*32 );
  this->addWaypoint( wp );
  return 0;
}

} // end of namespace
